<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Define the central API endpoint for Proshop Royal Admin Notices.
 * THIS MUST BE THE URL OF YOUR CENTRAL SERVER (e.g., iranavada.com)
 * where the Proshop Royal Admin Notices PLUGIN is installed and active.
 */
define( 'PROSHOP_ROYAL_CENTRAL_API_URL', 'https://iranavada.com/wp-json/irava-notice/v1/notices' ); 

/**
 * Define the updaterID for the 'test' environment.
 */
define( 'UPDATER_ID_TEST', 'test' ); 

function irava_display_admin_notices() { 
    // Only run in admin area
    if ( ! is_admin() ) {
        return;
    }

    // Determine the Proshop Royal version based on updaterID
    $client_version = '';
    if ( defined( 'updaterID' ) ) {
        switch ( updaterID ) {
            case 46800:
                $client_version = 'new-royal';
                break;
            case 23962:
                $client_version = 'old-royal';
                break;
            case 19977:
                $client_version = 'single-lang';
                break;
            case 19978:
                $client_version = 'multi-lang';
                break;
            case UPDATER_ID_TEST:
                $client_version = 'test';
                break;
            default:
                $client_version = ''; 
        }
    }

    // Transient key now includes client version and changed prefix
    $transient_key = 'irava_admin_notices_' . $client_version; 
    $notices_data = get_transient( $transient_key );
    
    if ( false === $notices_data ) {
        $api_url = PROSHOP_ROYAL_CENTRAL_API_URL;
        
        if ( ! empty( $client_version ) ) {
            $api_url = add_query_arg( 'version', $client_version, $api_url );
        }

        $response = wp_remote_get( $api_url, array(
            'timeout'     => 3, 
            'sslverify'   => true,
            'headers'     => array(
                'Accept' => 'application/json',
            ),
        ) );

        $notices_data = array(); 

        if ( is_wp_error( $response ) ) {
            error_log( 'Irava Admin Notices API Error (wp_remote_get): ' . $response->get_error_message() ); 
        } else {
            $http_code = wp_remote_retrieve_response_code( $response );
            $body = wp_remote_retrieve_body( $response );

            if ( $http_code === 200 ) {
                $decoded_data = json_decode( $body, true );

                if ( json_last_error() === JSON_ERROR_NONE && is_array( $decoded_data ) ) {
                    $valid_notices = array();
                    foreach ( $decoded_data as $item ) {
                        if ( is_array( $item ) 
                            && isset( $item['id'] ) && is_numeric( $item['id'] ) 
                            && isset( $item['message'] ) && is_string( $item['message'] ) 
                            && isset( $item['type'] ) && is_string( $item['type'] )
                            && isset( $item['can_dismiss_permanently'] ) && is_bool( $item['can_dismiss_permanently'] )
                        ) {
                            $allowed_types = array( 'info', 'success', 'error', 'warning' );
                            if ( in_array( $item['type'], $allowed_types ) ) {
                                $valid_notices[] = $item;
                            } else {
                                $valid_notices[] = array( 
                                    'id' => $item['id'], 
                                    'message' => $item['message'], 
                                    'type' => 'info', 
                                    'can_dismiss_permanently' => $item['can_dismiss_permanently']
                                );
                                error_log( 'Irava Admin Notices API Response: Invalid notice type provided, defaulted to info. Type: ' . $item['type'] ); 
                            }
                        } else {
                            error_log( 'Irava Admin Notices API Response: Invalid notice item format. Item: ' . print_r($item, true) ); 
                        }
                    }
                    if ( ! empty( $valid_notices ) ) {
                        $notices_data = $valid_notices;
                    } else {
                        error_log( 'Irava Admin Notices API Response: Valid JSON, but no valid formatted messages found. Response: ' . print_r($decoded_data, true) ); 
                    }
                } else {
                    error_log( 'Irava Admin Notices API Response: Invalid JSON or not an array. HTTP Code: ' . $http_code . ', Body: ' . $body ); 
                }
            } else {
                error_log( 'Irava Admin Notices API Response: Non-200 HTTP code. Code: ' . $http_code . ', Body: ' . $body ); 
            }
        }
        
        set_transient( $transient_key, $notices_data, 8 * HOUR_IN_SECONDS ); 
    }
    
    // Display notices. No user meta or localStorage check here as it's not permanently stored.
    if ( is_array( $notices_data ) && ! empty( $notices_data ) ) {
        foreach ( $notices_data as $notice_item ) {
            if ( is_array( $notice_item ) 
                && isset( $notice_item['id'] ) && isset( $notice_item['message'] ) 
                && isset( $notice_item['type'] ) && isset( $notice_item['can_dismiss_permanently'] ) 
            ) {
                $notice_id = $notice_item['id']; 
                $notice_message = $notice_item['message'];
                $notice_type = $notice_item['type'];
                $can_dismiss_permanently = $notice_item['can_dismiss_permanently']; 

                // Add a specific class for notices that can be permanently dismissed
                // Also, conditionally add the 'is-dismissible' class based on whether it's temporarily dismissible.
                $dismiss_class = '';
                if ($can_dismiss_permanently) {
                    $dismiss_class = 'irava-permanent-dismissible';
                } else {
                    $dismiss_class = 'is-dismissible'; 
                }
                
                // All notices are initially hidden by CSS and then shown by JS
                ?>
                <div class="notice notice-<?php echo esc_attr( $notice_type ); ?> irava-admin-notice <?php echo esc_attr($dismiss_class); ?>" data-notice-id="<?php echo esc_attr( $notice_id ); ?>" data-permanent-dismissible="<?php echo $can_dismiss_permanently ? 'true' : 'false'; ?>" style="display: none;">
                    <?php echo wp_kses_post( $notice_message ); ?>
                    <?php if ( $can_dismiss_permanently ) : ?>
                        <button type="button" class="irava-close-button">بستن دائم</button>
                    <?php endif; ?>
                </div>
                <?php
            } else {
                error_log( 'Irava Admin Notices Display Error: Malformed notice item during display. Item: ' . print_r($notice_item, true) ); 
            }
        }
    }
}
add_action( 'admin_notices', 'irava_display_admin_notices' ); 

/**
 * Add custom CSS and JavaScript for admin notices directly in the head.
 * This is loaded only in the admin area.
 */
function irava_admin_notice_custom_styles_and_scripts() { 
    // Only load in admin area
    if ( ! is_admin() ) {
        return;
    }
    ?>
    <style type="text/css">
        /* Custom styles for Irava Admin Notices */
        .irava-admin-notice { 
            border-radius: 5px;
            margin-bottom: 15px; 
            box-shadow: 0 2px 5px rgba(0,0,0,0.1);
            padding: 10px 15px; 
            line-height: 1.6; 
            position: relative; 
        }

        /* Standard WordPress dismiss button (X) for temporary dismissible notices */
        .irava-admin-notice.notice.is-dismissible .notice-dismiss:before {
            background: none;
            color: #72777c;
            font: normal 20px/1 'dashicons';
            content: '\f153';
            -webkit-font-smoothing: antialiased;
            -moz-osx-font-smoothing: grayscale;
            width: 20px;
            height: 20px;
            display: inline-block;
            vertical-align: top;
            cursor: pointer; 
        }
        .irava-admin-notice.notice.is-dismissible .notice-dismiss {
            float: right;
            padding: 9px 0 9px 9px;
            margin: 0;
            height: auto;
            border: none;
            background: none;
            color: #72777c;
            cursor: pointer;
            text-decoration: none;
            -webkit-appearance: none;
        }

        /* Hide default dismiss for permanent notices, as we add our own button */
        .irava-admin-notice.irava-permanent-dismissible .notice-dismiss {
            display: none !important; 
        }
        
        /* Custom "بستن دائم" button for permanent dismissible notices */
        .irava-admin-notice .irava-close-button {
            position: absolute;
            left: 6px;
            top: 6px;
            cursor: pointer;
            padding: 2px 10px;
            color: #555;
            font-size: 13px;
            text-decoration: none;
            background: #fff;
            border-radius: 5px;
            border: 1px solid #ddd;
        }
        .irava-admin-notice .irava-close-button:hover {
            background: #e91e63;
            border-color: #e91e63;
            color: #fff;
        }
        
        
        /* buttom link styles */
        a.irava_button {
            text-decoration: none!important;
            margin: 10px;
            background: #fff;
            padding: 5px 10px;
            border-radius: 5px;
            border: 1px solid #ddd;
            display: inline-block;
        }
        a.irava_button:hover {
            background: #008ccb;
            border-color: #008ccb;
            color: #fff!important;
        }
        a.irava_button:focus {
            box-shadow: 0 0 0;
        }
        
        
        /* Info Notice (Blue) */
        .notice.notice-info.irava-admin-notice { 
            background-color: #e0f2f7; 
            border-left-color: #0073aa; 
            color: #313131;
            padding: 10px 15px 10px 38px !important;
        }

        /* Success Notice (Green) */
        .notice.notice-success.irava-admin-notice { 
            background-color: #e6ffe6; 
            border-left-color: #46b450; 
            color: #313131;
            padding: 10px 15px 10px 38px !important;
        }

        /* Error Notice (Red) */
        .notice.notice-error.irava-admin-notice { 
            background-color: #ffe0e0; 
            border-left-color: #dc3232; 
            color: #313131;
            padding: 10px 15px 10px 38px !important;
        }

        /* Warning Notice (Yellow/Orange) */
        .notice.notice-warning.irava-admin-notice { 
            background-color: #fffacd; 
            border-left-color: #ffb900; 
            color: #313131;
            padding: 10px 15px 10px 38px !important;
        }

        .irava-admin-notice img { 
            max-width: 100%; 
            height: auto;
            display: block; 
            margin: 10px 0; 
            border-radius: 3px;
        }

        .irava-admin-notice a { 
            color: #0073aa; 
            text-decoration: underline;
        }

        .irava-admin-notice a:hover { 
            color: #005177;
        }
        
        .irava-admin-notice p:first-child { 
            margin-top: 0; 
        }
        .irava-admin-notice p:last-child { 
            margin-bottom: 0; 
        }
    </style>
    <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Key for localStorage to store dismissed notices with their expiry timestamp
            const dismissedNoticesKey = 'irava_dismissed_notices_expiry'; 
            //const twelveHours = 12 * 60 * 60 * 1000; // 12 hours in milliseconds
            const threeHundredDays = 300 * 24 * 60 * 60 * 1000; // 300 Days

            // Function to get dismissed notices and their expiry timestamps
            function getDismissedNoticesWithExpiry() {
                try {
                    var dismissedData = localStorage.getItem(dismissedNoticesKey);
                    // Filter out expired notices
                    var parsedData = dismissedData ? JSON.parse(dismissedData) : {};
                    var currentTimestamp = new Date().getTime();
                    var activeDismissed = {};
                    for (var noticeId in parsedData) {
                        if (parsedData.hasOwnProperty(noticeId) && parsedData[noticeId] > currentTimestamp) {
                            activeDismissed[noticeId] = parsedData[noticeId];
                        }
                    }
                    return activeDismissed;
                } catch (e) {
                    console.error("Irava Notices: Error reading from localStorage:", e);
                    return {};
                }
            }

            // Function to save a dismissed notice with its expiry timestamp
            function saveDismissedNoticeWithExpiry(noticeId) {
                var dismissedData = getDismissedNoticesWithExpiry(); // Get current active ones
                // var expiryTimestamp = new Date().getTime() + twelveHours;  // 12 hours 
                var expiryTimestamp = new Date().getTime() + threeHundredDays;  // 300 Daye
                dismissedData[noticeId] = expiryTimestamp;
                try {
                    localStorage.setItem(dismissedNoticesKey, JSON.stringify(dismissedData));
                } catch (e) {
                    console.error("Irava Notices: Error writing to localStorage:", e);
                }
            }

            // Hide all notices initially (handled by PHP style="display: none;")
            // Then, iterate through all notices and show/hide based on dismiss status and expiry
            var activeDismissedNotices = getDismissedNoticesWithExpiry();
            $('.irava-admin-notice').each(function() {
                var $notice = $(this);
                var noticeId = $notice.data('notice-id');
                var isPermanentDismissible = $notice.data('permanent-dismissible');
                
                // If it's a permanent dismissible notice AND its ID is in our active dismissed list
                if (isPermanentDismissible && activeDismissedNotices.hasOwnProperty(noticeId)) {
                    $notice.hide(); // Keep hidden
                } else {
                    $notice.show(); // Show it
                }
            });


            // Handle dismiss button click for permanent notices
            $(document).on('click', '.irava-admin-notice .irava-close-button', function(e) { 
                e.preventDefault(); 
                var $notice = $(this).closest('.irava-admin-notice'); 
                var noticeId = $notice.data('notice-id');
                
                if (noticeId) {
                    saveDismissedNoticeWithExpiry(noticeId); // Save with expiry
                }
                $notice.fadeOut(); 
            });
            
            // Handle standard WordPress dismiss button (X) for temporary notices
            $(document).on('click', '.irava-admin-notice.is-dismissible .notice-dismiss', function(e) {
                // Let WordPress's default behavior handle this.
                // It usually fades out and removes the notice from the DOM for the session.
            });

            // Removed all AJAX migration/saving logic as user meta is no longer used for this feature.
        });
    </script>
    <?php
}
add_action( 'admin_head', 'irava_admin_notice_custom_styles_and_scripts' ); 

// AJAX handler for saving dismissed notices to user meta (REMOVED)
// This function and its add_action hook should be REMOVED from your functions.php as well.
// function irava_save_dismissed_notices_to_user_meta() { /* ... */ }
// remove_action( 'wp_ajax_irava_save_dismissed_notices_meta', 'irava_save_dismissed_notices_to_user_meta' );