document.addEventListener("DOMContentLoaded", function () {

    const availableAttributesList = document.getElementById('availableAttributesList');
    const selectedAttributesList = document.getElementById('selectedAttributesList');
    const selectedAttributesInput = document.getElementById('selected_attributes_input');

    let draggedItem = null;

    // تنظیم درگ و دراپ برای آیتم‌های موجود
    function setupDragAndDropForExistingItems() {
        const existingItems = selectedAttributesList.querySelectorAll('li');
        existingItems.forEach(item => {
            setupDragAndDropForNewItem(item); // اعمال رویدادهای درگ و دراپ به آیتم‌های موجود
        });
    }

    // افزودن پیام‌های پیش‌فرض
    const noAttributesAvailableMessage = document.createElement('span');
    noAttributesAvailableMessage.textContent = 'تمامی موارد انتخاب شده است.';
    noAttributesAvailableMessage.style.display = 'none'; // مخفی به‌صورت پیش‌فرض
    noAttributesAvailableMessage.classList.add('no-attributes-message');
    availableAttributesList.parentElement.appendChild(noAttributesAvailableMessage);

    const noAttributesSelectedMessage = document.createElement('span');
    noAttributesSelectedMessage.textContent = 'هیچ ویژگی‌ای انتخاب نشده است.';
    noAttributesSelectedMessage.style.display = 'none'; // مخفی به‌صورت پیش‌فرض
    noAttributesSelectedMessage.classList.add('no-attributes-message');
    selectedAttributesList.parentElement.appendChild(noAttributesSelectedMessage);

    // اضافه کردن رویداد کلیک به آیتم‌های لیست موجود
    availableAttributesList.addEventListener('click', function (event) {
        if (event.target.tagName === 'LI') {
            const li = event.target;

            // انتقال به لیست ویژگی‌های انتخاب‌شده
            li.remove();
            const newLabel = li.textContent.trim().replace('⋮⋮', ''); // حذف ۶ نقطه درگ
            const newLi = createSelectedItem(li.dataset.id, newLabel);
            selectedAttributesList.appendChild(newLi);

            updateSelectedAttributesInput();
            checkEmptyLists();

            // اعمال رویدادهای درگ و دراپ برای آیتم جدید
            setupDragAndDropForNewItem(newLi);
        }
    });

    selectedAttributesList.addEventListener('click', function (event) {
        if (event.target.tagName === 'BUTTON' && event.target.classList.contains('remove')) {
            const li = event.target.closest('li');
    
            // انتقال به لیست ویژگی‌های موجود
            li.remove();
    
            // استفاده از querySelector برای دریافت صحیح عنوان ویژگی
            const newLabel = li.querySelector('.attribute-text .attribute-label').textContent.trim(); // اصلاح این بخش
            const newLi = createAvailableItem(li.dataset.id, newLabel);
            availableAttributesList.appendChild(newLi);
    
            updateSelectedAttributesInput();
            checkEmptyLists();
        }
    });

    // به‌روزرسانی مقدار فیلد مخفی
    function updateSelectedAttributesInput() {
        const selectedIds = Array.from(selectedAttributesList.querySelectorAll('li')).map(li => li.dataset.id);
        selectedAttributesInput.value = selectedIds.join(',');
    }

    // ایجاد آیتم جدید برای لیست انتخاب‌شده
    function createSelectedItem(id, label) {
        const li = document.createElement('li');
        li.dataset.id = id;

        // ساخت ساختار span مشابه به کد PHP
        const span = document.createElement('span');
        span.classList.add('attribute-text');
        
        const dragHandle = document.createElement('span');
        dragHandle.classList.add('drag-handle');
        dragHandle.textContent = '⋮⋮';

        // اضافه کردن handle به span
        span.appendChild(dragHandle);

        // اضافه کردن label به span
        const labelSpan = document.createElement('span');
        labelSpan.classList.add('attribute-label');
        labelSpan.textContent = label; // فقط نمایش label بدون ۶ نقطه
        span.appendChild(labelSpan);

        li.appendChild(span);

        const deleteButton = document.createElement('button');
        deleteButton.textContent = 'حذف';
        deleteButton.classList.add('remove');
        li.appendChild(deleteButton);

        return li;
    }

    // ایجاد آیتم جدید برای لیست موجود (بدون drag-handle)
    function createAvailableItem(id, label) {
        const li = document.createElement('li');
        li.dataset.id = id;

        // تنها نام ویژگی را اضافه می‌کنیم، بدون drag-handle
        li.textContent = label; // فقط نمایش label بدون ۶ نقطه

        return li;
    }
    

    // بررسی خالی بودن لیست‌ها
    function checkEmptyLists() {
        if (availableAttributesList.childElementCount === 0) {
            noAttributesAvailableMessage.style.display = 'block';
        } else {
            noAttributesAvailableMessage.style.display = 'none';
        }

        if (selectedAttributesList.childElementCount === 0) {
            noAttributesSelectedMessage.style.display = 'block';
        } else {
            noAttributesSelectedMessage.style.display = 'none';
        }
    }

    // رویدادهای Drag and Drop برای آیتم‌های جدید
    function setupDragAndDropForNewItem(newItem) {
        newItem.setAttribute('draggable', 'true');

        newItem.addEventListener('dragstart', function (event) {
            draggedItem = event.target;
            event.target.classList.add('dragging');
        });

        newItem.addEventListener('dragend', function (event) {
            event.target.classList.remove('dragging');
            updateSelectedAttributesInput();
        });

        selectedAttributesList.addEventListener('dragover', function (event) {
            event.preventDefault();
            const afterElement = getDragAfterElement(selectedAttributesList, event.clientY);
            if (afterElement == null) {
                selectedAttributesList.appendChild(draggedItem);
            } else {
                selectedAttributesList.insertBefore(draggedItem, afterElement);
            }
        });
    }

    // پیدا کردن آیتم بعدی بر اساس موقعیت موس
    function getDragAfterElement(container, y) {
        const draggableElements = [...container.querySelectorAll('li:not(.dragging)')];

        return draggableElements.reduce((closest, child) => {
            const box = child.getBoundingClientRect();
            const offset = y - box.top - box.height / 2;
            if (offset < 0 && offset > closest.offset) {
                return { offset: offset, element: child };
            } else {
                return closest;
            }
        }, { offset: Number.NEGATIVE_INFINITY }).element;
    }

    // بررسی اولیه لیست‌ها
    checkEmptyLists();
    setupDragAndDropForExistingItems(); // اعمال درگ و دراپ برای آیتم‌های از پیش موجود

});
