<?php
/**
 * ابزار بررسی اتصال لایسنس قالب وردپرس
 * این کد به شما کمک می‌کند تا مشکلات اتصال بین کلاینت و سرور لایسنس را تشخیص دهید.
 * 
 * نحوه استفاده: این فایل را در پوشه قالب فرزند قرار دهید و از طریق مرورگر به آن دسترسی پیدا کنید:
 * https://your-domain.com/wp-content/themes/your-child-theme/license-check.php
 */

// تنظیمات سرورهای لایسنس
$license_servers = [
    [
        'name' => 'سرور اصلی',
        'domain' => 'iranavada.com',
        'license_endpoint' => 'https://iranavada.com/wp-json/license/validate',
    ],
    [
        'name' => 'سرور پشتیبان',
        'domain' => 'royal-studio.ir',
        'license_endpoint' => 'https://royal-studio.ir/wp-json/wp/v2/proshoproyal_studio_rest',
    ]
];

// اطلاعات سیستم (بدون نمایش IP)
$system_info = [
    'php_version' => PHP_VERSION,
    'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'نامشخص',
    'host_domain' => $_SERVER['HTTP_HOST'] ?? 'نامشخص',
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'نامشخص',
    'allow_url_fopen' => ini_get('allow_url_fopen') ? 'فعال' : 'غیرفعال',
    'curl_enabled' => function_exists('curl_version') ? 'فعال' : 'غیرفعال',
    'time' => date('Y-m-d H:i:s'),
];

// مشخصات شبکه کلاینت
$client_network = [
    'dns_works' => function_exists('gethostbyname') ? 'فعال' : 'غیرفعال',
    'outbound_internet' => false,
    'ssl_works' => false,
    'server_connections' => []
];

// تست اتصال به یک سرویس اینترنتی معتبر
$test_dns = @gethostbyname('google.com');
$client_network['outbound_internet'] = ($test_dns != 'google.com');

// تست اتصال SSL
if (function_exists('curl_version')) {
    $ch = curl_init('https://www.google.com');
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    curl_setopt($ch, CURLOPT_NOBODY, true);
    $result = curl_exec($ch);
    $client_network['ssl_works'] = ($result !== false);
    curl_close($ch);
}

// تست اتصال به سرورهای لایسنس
foreach ($license_servers as $server) {
    $start_time = microtime(true);
    $connection_test = [
        'server' => $server['name'],
        'domain' => $server['domain'],
        'can_resolve' => false,
        'can_connect' => false,
        'can_connect_api' => false,
        'response_time' => 0,
        'errors' => []
    ];

    // آیا دامنه قابل تبدیل به IP است؟
    $server_ip = @gethostbyname($server['domain']);
    if ($server_ip != $server['domain']) {
        $connection_test['can_resolve'] = true;
        
        // آیا می‌توان به پورت 80 متصل شد؟
        $socket = @fsockopen($server['domain'], 80, $errno, $errstr, 10);
        if (is_resource($socket)) {
            $connection_test['can_connect'] = true;
            fclose($socket);
        } else {
            $connection_test['errors'][] = "خطای اتصال به پورت 80: [$errno] $errstr";
        }
        
        // آیا می‌توان به API لایسنس دسترسی داشت؟
        if (function_exists('curl_version')) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $server['license_endpoint']);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
            curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
            curl_setopt($ch, CURLOPT_TIMEOUT, 15);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
            curl_setopt($ch, CURLOPT_NOBODY, true);
            
            // ارسال یک پارامتر تست برای API
            curl_setopt($ch, CURLOPT_POSTFIELDS, ['action' => 'ping_test']);
            
            $result = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            
            if ($result !== false) {
                $connection_test['can_connect_api'] = true;
                $connection_test['api_http_code'] = $http_code;
            } else {
                $connection_test['errors'][] = "خطای دسترسی به API: " . curl_error($ch);
            }
            
            curl_close($ch);
        } else {
            $connection_test['errors'][] = "CURL نصب نشده است و نمی‌توان اتصال API را تست کرد.";
        }
    } else {
        $connection_test['errors'][] = "عدم توانایی در تبدیل دامنه به IP";
    }
    
    // محاسبه زمان پاسخگویی
    $connection_test['response_time'] = round((microtime(true) - $start_time) * 1000, 2) . " میلی‌ثانیه";
    
    // افزودن نتایج به آرایه کلی
    $client_network['server_connections'][] = $connection_test;
}

// ایجاد لاگ محلی برای تحلیل بعدی
$child_theme_dir = dirname(__FILE__); // مسیر پوشه قالب فرزند
$log_file = 'license_connection_log_' . date('Y-m-d_H-i-s') . '.txt';
$log_path = $child_theme_dir . '/' . $log_file;

$log_data = "=== گزارش بررسی اتصال لایسنس قالب ===\n\n";
$log_data .= "--- اطلاعات سیستم ---\n";
foreach ($system_info as $key => $value) {
    $log_data .= "$key: $value\n";
}

$log_data .= "\n--- وضعیت شبکه ---\n";
$log_data .= "DNS کار می‌کند: " . $client_network['dns_works'] . "\n";
$log_data .= "اتصال به اینترنت: " . ($client_network['outbound_internet'] ? 'موفق' : 'ناموفق') . "\n";
$log_data .= "اتصال SSL: " . ($client_network['ssl_works'] ? 'موفق' : 'ناموفق') . "\n";

$log_data .= "\n--- نتایج اتصال به سرورها ---\n";
foreach ($client_network['server_connections'] as $test) {
    $log_data .= "\nسرور: {$test['server']} ({$test['domain']})\n";
    $log_data .= "تبدیل نام به IP: " . ($test['can_resolve'] ? 'موفق' : 'ناموفق') . "\n";
    $log_data .= "اتصال به پورت 80: " . ($test['can_connect'] ? 'موفق' : 'ناموفق') . "\n";
    $log_data .= "اتصال به API: " . ($test['can_connect_api'] ? 'موفق' : 'ناموفق') . "\n";
    $log_data .= "زمان پاسخگویی: {$test['response_time']}\n";
    
    if (!empty($test['errors'])) {
        $log_data .= "خطاها:\n";
        foreach ($test['errors'] as $error) {
            $log_data .= " - $error\n";
        }
    }
}

// ذخیره لاگ در سرور
file_put_contents($log_path, $log_data);

// تست شبیه‌سازی اتصال لایسنس برای دریافت خطاهای احتمالی
function test_license_connection($server_endpoint, $license_key = 'test-license-key') {
    if (!function_exists('curl_version')) {
        return [
            'success' => false,
            'message' => 'CURL نصب نشده است.'
        ];
    }
    
    $post_data = [
        'license_key' => $license_key,
        'domain' => $_SERVER['HTTP_HOST'] ?? 'unknown',
        'action' => 'validate'
    ];
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $server_endpoint);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($post_data));
    curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, true);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    $error_no = curl_errno($ch);
    
    curl_close($ch);
    
    return [
        'success' => ($response !== false),
        'http_code' => $http_code,
        'response' => $response,
        'error' => $error,
        'error_no' => $error_no
    ];
}

// شبیه‌سازی بررسی لایسنس برای هر سرور
$license_tests = [];
foreach ($license_servers as $server) {
    $license_tests[$server['name']] = test_license_connection($server['license_endpoint']);
}

// آدرس نسبی فایل لاگ برای دانلود
$log_rel_path = basename($log_file);

// گزارش نهایی به صورت HTML برای نمایش
?>
<!DOCTYPE html>
<html dir="rtl" lang="fa-IR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>گزارش بررسی اتصال لایسنس</title>
    <style>
        body {
            font-family: Tahoma, Arial, sans-serif;
            line-height: 1.6;
            margin: 20px;
            background-color: #f5f5f5;
        }
        h1, h2, h3 {
            color: #333;
        }
        .container {
            max-width: 800px;
            margin: 0 auto;
            background: white;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }
        .success {
            color: green;
            font-weight: bold;
        }
        .error {
            color: red;
            font-weight: bold;
        }
        .warning {
            color: orange;
            font-weight: bold;
        }
        .box {
            border: 1px solid #ddd;
            padding: 15px;
            margin-bottom: 20px;
            border-radius: 5px;
        }
        table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
        }
        th, td {
            border: 1px solid #ddd;
            padding: 8px;
            text-align: right;
        }
        th {
            background-color: #f2f2f2;
        }
        .recommendations {
            background-color: #e9f7ef;
            padding: 15px;
            border-radius: 5px;
            margin-top: 20px;
        }
        .download-link {
            display: inline-block;
            margin: 20px 0;
            padding: 10px 15px;
            background-color: #4CAF50;
            color: white;
            text-decoration: none;
            border-radius: 5px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>گزارش بررسی اتصال به سرور قالب</h1>
        
        <div class="box">
            <h2>اطلاعات سیستم</h2>
            <table>
                <?php foreach ($system_info as $key => $value): ?>
                <tr>
                    <th><?php echo htmlspecialchars($key); ?></th>
                    <td><?php echo htmlspecialchars($value); ?></td>
                </tr>
                <?php endforeach; ?>
            </table>
        </div>
        
        <div class="box">
            <h2>وضعیت شبکه</h2>
            <table>
                <tr>
                    <th>DNS کار می‌کند</th>
                    <td class="<?php echo $client_network['dns_works'] == 'فعال' ? 'success' : 'error'; ?>">
                        <?php echo $client_network['dns_works']; ?>
                    </td>
                </tr>
                <tr>
                    <th>اتصال به اینترنت</th>
                    <td class="<?php echo $client_network['outbound_internet'] ? 'success' : 'error'; ?>">
                        <?php echo $client_network['outbound_internet'] ? 'موفق' : 'ناموفق'; ?>
                    </td>
                </tr>
                <tr>
                    <th>اتصال SSL</th>
                    <td class="<?php echo $client_network['ssl_works'] ? 'success' : 'error'; ?>">
                        <?php echo $client_network['ssl_works'] ? 'موفق' : 'ناموفق'; ?>
                    </td>
                </tr>
            </table>
        </div>
        
        <div class="box">
            <h2>نتایج اتصال به سرورها</h2>
            <?php foreach ($client_network['server_connections'] as $test): ?>
            <h3><?php echo htmlspecialchars($test['server']); ?> (<?php echo htmlspecialchars($test['domain']); ?>)</h3>
            <table>
                <tr>
                    <th>تبدیل نام به IP</th>
                    <td class="<?php echo $test['can_resolve'] ? 'success' : 'error'; ?>">
                        <?php echo $test['can_resolve'] ? 'موفق' : 'ناموفق'; ?>
                    </td>
                </tr>
                <tr>
                    <th>اتصال به پورت 80</th>
                    <td class="<?php echo $test['can_connect'] ? 'success' : 'error'; ?>">
                        <?php echo $test['can_connect'] ? 'موفق' : 'ناموفق'; ?>
                    </td>
                </tr>
                <tr>
                    <th>اتصال به API</th>
                    <td class="<?php echo $test['can_connect_api'] ? 'success' : 'error'; ?>">
                        <?php echo $test['can_connect_api'] ? 'موفق' : 'ناموفق'; ?>
                        <?php if (isset($test['api_http_code'])): ?>
                        <br>کد HTTP: <?php echo htmlspecialchars($test['api_http_code']); ?>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <th>زمان پاسخگویی</th>
                    <td><?php echo htmlspecialchars($test['response_time']); ?></td>
                </tr>
                <?php if (!empty($test['errors'])): ?>
                <tr>
                    <th>خطاها</th>
                    <td class="error">
                        <ul>
                            <?php foreach ($test['errors'] as $error): ?>
                            <li><?php echo htmlspecialchars($error); ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </td>
                </tr>
                <?php endif; ?>
            </table>
            <?php endforeach; ?>
        </div>
        
        <div class="box">
            <h2>شبیه‌سازی فعال‌سازی لایسنس</h2>
            <?php foreach ($license_tests as $server_name => $test): ?>
            <h3><?php echo htmlspecialchars($server_name); ?></h3>
            <table>
                <tr>
                    <th>وضعیت اتصال</th>
                    <td class="<?php echo $test['success'] ? 'success' : 'error'; ?>">
                        <?php echo $test['success'] ? 'موفق' : 'ناموفق'; ?>
                    </td>
                </tr>
                <tr>
                    <th>کد HTTP</th>
                    <td>
                        <?php echo isset($test['http_code']) ? htmlspecialchars($test['http_code']) : 'نامشخص'; ?>
                    </td>
                </tr>
                <?php if (!$test['success']): ?>
                <tr>
                    <th>خطای CURL</th>
                    <td class="error">
                        <?php echo htmlspecialchars($test['error']); ?> (<?php echo htmlspecialchars($test['error_no']); ?>)
                    </td>
                </tr>
                <?php endif; ?>
            </table>
            <?php endforeach; ?>
        </div>
        
        <div class="recommendations">
            <h2>توصیه‌ها و راه‌حل‌ها</h2>
            <ul>
                <?php if (!$client_network['dns_works']): ?>
                <li>مشکل در سیستم DNS وجود دارد. با ارائه دهنده خدمات میزبانی خود تماس بگیرید.</li>
                <?php endif; ?>
                
                <?php if (!$client_network['outbound_internet']): ?>
                <li>اتصال به اینترنت مشکل دارد. فایروال یا تنظیمات امنیتی سرور را بررسی کنید.</li>
                <?php endif; ?>
                
                <?php if (!$client_network['ssl_works']): ?>
                <li>اتصال SSL با مشکل مواجه است. نسخه PHP و تنظیمات SSL را بررسی کنید.</li>
                <?php endif; ?>
                
                <?php 
                $has_server_connection_issue = false;
                foreach ($client_network['server_connections'] as $test) {
                    if (!$test['can_connect_api']) {
                        $has_server_connection_issue = true;
                    }
                }
                if ($has_server_connection_issue): 
                ?>
                <li>مشکل در اتصال به API لایسنس وجود دارد. موارد زیر را بررسی کنید:
                    <ul>
                        <li>آیا فایروال سرور شما ارتباط خروجی را مسدود می‌کند؟</li>
                        <li>آیا تنظیمات <code>allow_url_fopen</code> یا <code>CURL</code> در PHP فعال است؟</li>
                        <li>آیا بسته امنیتی مانند mod_security درخواست‌ها را مسدود می‌کند؟</li>
                    </ul>
                </li>
                <?php endif; ?>
                
                <li>در صورت استفاده از لایسنس اصلی، با پشتیبانی قالب تماس بگیرید و این گزارش را برای آنها ارسال کنید.</li>
                <li>اگر از چند سایت روی یک هاست استفاده می‌کنید، بررسی کنید که تعداد لایسنس‌های شما کافی باشد.</li>
            </ul>
        </div>
        
        <a href="<?php echo $log_rel_path; ?>" class="download-link" download>دانلود فایل لاگ</a>
    </div>
</body>
</html>
